from threading import Thread
from queue import Queue

import time, numpy as np, cv2
        
class Camera(object):
    def __init__(self,_id,video_path,encoder=None, width=1280, height=720, framerate=30, log=None, maxsize=10 , use_gstr=True):
        self.ID = _id
        self.__log = self.__log if log is None else log
        self.__isCaptured = False
        self.__frame = None
        self.isOpened = False
        self.video_path = video_path
        self.width = width
        self.height = height
        self.framerate = framerate
        self.encoder = encoder
        self.use_gstr = use_gstr
        self.__thread = Thread(target=self.__job)
        self.resultQueue = Queue(maxsize=maxsize)
    def start(self):
        self.__isCaptured = False
        self.__frame = None
        self.isOpened = True
        self.__thread.start()
    def stop(self):
        self.__isCaptured = False
        self.__frame = None
        self.isOpened = False
    def capture(self):
        return self.__isCaptured, self.__frame
    def __job(self):
        w = self.width
        h = self.height
        fps = self.framerate
        if self.use_gstr:
            source = self.__gstreamer(self.video_path,w, h, fps)
        while self.isOpened:
            try:
                if self.use_gstr:
                    print("Using GSTREAMER")
                    print(source)
                    dev = cv2.VideoCapture(source, cv2.CAP_GSTREAMER)
                else:
                    print("Not Using GSTREAMER")
                    dev = cv2.VideoCapture(self.video_path)
                while self.isOpened:
                    if dev.isOpened():
                        ret, frame = dev.read()
                        if not ret:
                            break
                        self.__frame = frame
                        self.__isCaptured = ret
                        # 滿了先丟掉一個再放入
                        if(self.resultQueue.full()):
                            # 捨棄
                            _=self.resultQueue.get_nowait()
                        self.resultQueue.put((self.__isCaptured,self.__frame))
                        time.sleep(1 / fps)
            except Exception as e:
                raise (e)
                self.__isCaptured = False
                self.__frame = None
                self.__log("camera stop with error")
        dev.release()
        self.__isCaptured = False
        self.__frame = None
        self.__log("camera stop")
        
    def __gstreamer(self, source=0,width=1280, height=720, framerate=30, flip_method=2):        
        if source==0:
            # Camera Device Stream
            return (
              f'nvarguscamerasrc ! video/x-raw(memory:NVMM), ' +
              f'width=(int){width}, height=(int){height}, ' +
              f'format=(string)NV12, framerate=(fraction){framerate}/1 ! ' +
              f'nvvidconv flip-method={flip_method} ! ' +
              f'video/x-raw, width=(int){width}, height=(int){height}, format=(string)BGRx ! ' +
              'videoconvert ! video/x-raw, format=(string)BGR ! appsink')
        elif source.endswith(".mp4") or source.endswith(".MOV") or source.endswith(".avi"):
            return (
                f'filesrc location={source} ! '+
                'qtdemux ! queue ! h264parse ! nvv4l2decoder ! nvvidconv ! video/x-raw,format=BGRx ! queue ! videoconvert ! queue ! video/x-raw, format=BGR ! appsink')
        else:
            # Rtsp stream
            if self.encoder=="h265":
                print("h265")
                return (
                   f'rtspsrc location={source} ! ' +
#                    'watchdog timeout=10000 !'+
                   'rtph265depay ! h265parse ! nvv4l2decoder ! nvvidconv ! '+
                   'video/x-raw,format=BGRx ! videoconvert ! video/x-raw,format=BGR ! appsink'
               )
            elif self.encoder=="h264":
                print("h264")
                gst_str = ('rtspsrc location={} latency={} ! '
                   'rtph264depay ! h264parse ! omxh264dec ! '
                   'nvvidconv ! '
                   'video/x-raw, width=(int){}, height=(int){}, '
                   'format=(string)BGRx ! '
                   'videoconvert ! appsink').format(source, framerate, width, height)
                return gst_str
            else:
                assert False,"Please specifice encoder while parsing a RTSP stream (h265 or h264)"

    def __log(self, message):
        print(message)
        
    def __del__(self):
        self.stop()
