﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
// Program:
//     建立 iCatch DVR 歷史資料備份的執行緒物件。從 SDK 取得歷史影像與時間戳記後，根據戳記自動補齊缺少的影像
//     ，由於技術問題(無法將圖像已 H.264 的格式寫入影像檔中，因此將影像以片斷的方式儲存，取得完畢後，將補齊的
//     影像與歷史影像合併為單一檔案。
// History:
// 2019/05/31
namespace DVRModelV2
{
    public class DVRChannel
    {
        private Queue<KeyValuePair<DateTime, byte[]>> frames = new Queue<KeyValuePair<DateTime, byte[]>>();

        private Process process = new Process();

        private FontStyle style = FontStyle.Regular;

        private string familyName  = "Consolas";

        private float emSize = 12;

        private int count = -1;

        public int ID { get; set; }

        public int FPS { get; set; }

        public int Width { get; set; }

        public int Height { get; set; }

        public int Duration { get; set; }

        public string FontName { get => familyName; set => familyName = value; }

        public float FontSize { get => emSize; set => emSize = value; }

        public DateTime StartTime { get; set; }

        public void Start(string path)
        {
            Thread t = new Thread(Run);

            t.Start(path);
        }

        public void Record(byte[] buffer, DateTime timestamp)
        {
            KeyValuePair<DateTime, byte[]> pair = new KeyValuePair<DateTime, byte[]>(timestamp, buffer);
            // 將歷史影像與時間戳記寫入佇列中，等待寫入檔案
            frames.Enqueue(pair);
        }

        private int NewProcess(string filename, int count)
        {
            if (process.StartInfo.FileName != "" && !process.HasExited)
            {
                process.StandardInput.BaseStream.Close();
            }

            process = new Process();

            process.StartInfo.FileName = "lib" + Path.DirectorySeparatorChar + "ffmpeg.exe";

            process.StartInfo.Arguments = $"-framerate {FPS} -i - -c:v libx264 -r {FPS} \"{filename}.{count}.mp4\"";

            process.StartInfo.UseShellExecute = !(process.StartInfo.CreateNoWindow = process.StartInfo.RedirectStandardInput = true);

            return process.Start() ? count : count - 1;
        }

        private void Run(object path)
        {
            // 檢查資料夾路徑最後是否有包含分隔字元
            bool with = Convert.ToString(path).EndsWith(Convert.ToString(Path.DirectorySeparatorChar));
            // data\192.168.5.198\
            path = with ? path : Convert.ToString(path) + Path.DirectorySeparatorChar;
            // data\{host}\[CH00] 2019-05-02 12.00.00
            path = path + "[CH" + Convert.ToString(ID + 1).PadLeft(2, '0') + "] " + StartTime.ToString("yyyy-MM-dd HH.mm.ss");


            // 依照影像調整時間戳記的字型大小
            while (Width == 0 || Height == 0) { Thread.Sleep(1000 / FPS); }

            Graphics g = Graphics.FromImage(new Bitmap(Width, Height));

            Font f = new Font(familyName, emSize, style);

            SizeF s = g.MeasureString(StartTime.ToString("yyyy-MM-dd HH.mm.ss"), f);

            while (Height / 20 > s.Height)
            {
                f = new Font(familyName, emSize++, style);

                s = g.MeasureString(StartTime.ToString("yyyy-MM-dd HH.mm.ss"), f);
            }

            while (Height / 20 < s.Height)
            {
                f = new Font(familyName, emSize--, style);

                s = g.MeasureString(StartTime.ToString("yyyy-MM-dd HH.mm.ss"), f);
            }

            g.Dispose();

            // 於執行緒中記錄進度的物件
            DateTime time = StartTime;

            bool no_motion = false;
            // 當進度未到達前，此執行緒將持續執行
            while (StartTime.AddMinutes(Duration) > time)
            {
                if (frames.Count > 0 && frames.Peek().Key == time)
                {
                    if (no_motion)
                    {
                        count = NewProcess(path as string, count + 1);

                        no_motion = false;
                    }
                    
                    try
                    {
                        process.StandardInput.BaseStream.Write(frames.Peek().Value, 0, frames.Peek().Value.Length);

                        process.StandardInput.BaseStream.Flush();

                        DateTime dt1 = frames.Dequeue().Key; // 當前

                        DateTime dt2 = frames.Peek().Key; // 下一張

                        time = dt1.AddSeconds(1) != dt2 ? dt1 : dt2;

                        Console.WriteLine($"Channel: {ID} \t Buffer: {frames.Peek().Value.Length} \t Time: {time}");
                    }
                    catch { count = NewProcess(path as string, count + 1); }
                }
                while ((frames.Count > 0 && frames.Peek().Key > time) || (frames.Count == 0 && StartTime.AddMinutes(Duration) > time))
                {
                    if (!no_motion)
                    {
                        count = NewProcess(path as string, count + 1);

                        no_motion = true;
                    }

                    for (int i = 0; i < FPS; i++)
                    {
                        MemoryStream m = new MemoryStream();

                        Bitmap b = new Bitmap(Width, Height);

                        g = Graphics.FromImage(b);

                        g.Clear(Color.Black);

                        g.DrawString(time.ToString("yyyy-MM-dd HH.mm.ss"), f, Brushes.White, 0, Height / 20);

                        g.Save();

                        b.Save(m, ImageFormat.Jpeg);

                        try
                        {
                            process.StandardInput.BaseStream.Write(m.GetBuffer(), 0, m.GetBuffer().Length);

                            process.StandardInput.BaseStream.Flush();

                            Console.WriteLine($"Channel: {ID} \t Buffer: {m.GetBuffer().Length} \t Time: {time}");
                        }
                        catch { count = NewProcess(path as string, count + 1); }
                    }

                    no_motion = true;

                    //frameIndex = FPS;

                    time = time.AddSeconds(1);
                }
            }
            process.StandardInput.BaseStream.Close();
            
            
            // 合併影片檔 - 建立檔案清單
            FileStream file = new FileStream($"{path}.txt", FileMode.Create);

            StreamWriter sw = new StreamWriter(file);

            for (int i = 0; i <= count; i++)
            {
                bool isReady = false;

                while (!isReady)
                {
                    try
                    {
                        if (File.Exists($"{path}.{i}.mp4"))
                        {
                            isReady = File.Open($"{path}.{i}.mp4", FileMode.Open).Length > 0;

                            sw.WriteLine($"file \'{path}.{i}.mp4\'");

                            sw.Flush();
                        }
                        else { isReady = true; }
                    }
                    catch
                    {
                        isReady = false;

                        Console.WriteLine($"Channel: {ID} \t {path}.{i}.mp4 \t is locked.");
                    }
                    Thread.Sleep(1000 / FPS);
                }
            }

            sw.Close();


            // 合併影片檔 - 檢查影像片段
            if (File.Exists($"{path}.mp4")) { File.Delete($"{path}.mp4"); }

            if (process.StartInfo.FileName != "" && !process.HasExited)
            {
                process.StandardInput.BaseStream.Close();
            }

            process = new Process();

            process.StartInfo.FileName = "lib/ffmpeg.exe";

            process.StartInfo.Arguments = $"-f concat -safe 0 -i \"{path}.txt\" -c:v libx264 -r {FPS} \"{path}.mp4\"";

            process.StartInfo.UseShellExecute = !(process.StartInfo.CreateNoWindow = true);

            process.Start();

            while (!process.HasExited) { process.WaitForExit(); }

            for (int i = 0; i <= count; i++)
            {
                while (File.Exists($"{path}.{i}.mp4"))
                {
                    try { File.Delete($"{path}.{i}.mp4"); }
                    catch { };
                }
            }
            // 備份完成，紀錄進度作為下一個備份檔的開始時間
            StartTime = time;
        }
    }
}
