﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace DVRModel
{
    public class DVRChannel
    {
        private Process process = null;                                  // FFmpeg 外部執行緒

        private Queue<byte[]> frames = new Queue<byte[]>();              // 暫存的影像幀

        private MemoryStream no_motion = new MemoryStream();             // 無錄影的替代影像

        private Font font = new Font("Consolas", 12, FontStyle.Regular); // 預設的字型格式

        public int ID { get; set; }                  // 攝影機編號

        public int FPS { get; set; }                 // 攝影機固定幀率 (每秒多少幀)

        public string Filename { get; set; }         // 備份檔案名稱

        public int Width { get; set; }               // 攝影機影像寬度

        public int Height { get; set; }              // 攝影機影像高度

        public DateTime Time { get; set; }           // 最後備份的時間點 + 1s

        public bool Start() // [CH01] 2019-04-19 00.00.00.mp4
        {
            if (process == null)
            {
                FileStream stream = new FileStream("lib/NoMotion.jpg", FileMode.Open);

                stream.CopyTo(no_motion);

                stream.Close();

                Directory.CreateDirectory(Filename = Filename + Time.ToString("yyyy-MM-dd") + Path.DirectorySeparatorChar);

                process = new Process()
                {
                    StartInfo =
                    {
                        FileName = "lib/ffmpeg.exe",
                        Arguments = "-i - -c copy " + Filename + "[CH" + Convert.ToString(ID + 1).PadLeft(2, '0') + "]_" + Time.ToString("yyyy-MM-dd_HH.mm.ss") + ".mp4",
                        UseShellExecute = false,
                        CreateNoWindow = true,
                        RedirectStandardInput = true,
                    }
                };

                process.Start();

                return true;
            }
            else
                return false;
        }

        public void Enqueue(byte[] buffer)
        {
            frames.Enqueue(buffer);
        }

        public void Dequeue() // 每秒寫入一次
        {
            Bitmap b = new Bitmap(Width, Height);               // 建立空白影像物件

            Graphics g = Graphics.FromImage(b);                 // 建立畫布物件

            string text = Time.ToString("yyyy/MM/dd HH:mm:ss"); // 取得當前時間

            SizeF size = g.MeasureString(text, font);           // 取得時間字串於畫布中所使用的大小

            g.InterpolationMode = InterpolationMode.High;

            g.SmoothingMode = SmoothingMode.HighQuality;

            g.DrawImage(new Bitmap(no_motion), new Point(0, 0)); // 將 no motion 的影像繪製於畫布中

            no_motion.Position = 0; // 重設讀取指標位置

            while (Height / 20 > size.Height) // 放大時間字串
            {
                font = new Font(font.FontFamily, font.Size + 1, font.Style);

                size = g.MeasureString(text, font);
            }

            while (Height / 20 < size.Height) // 縮小時間字串
            {
                font = new Font(font.FontFamily, font.Size - 1, font.Style);

                size = g.MeasureString(text, font);
            }

            g.DrawString(text, font, Brushes.Red, new Point(0, 0)); // 將時間字串繪製於畫布中

            g.Save(); // 將變更儲存於影像物件

            byte[] frame = new byte[0];

            for (int i = 0; i < FPS; i++)
            {
                if (frames.Count > i)
                {
                    frame = frames.Dequeue();

                    process.StandardInput.BaseStream.Write(frame, 0, frame.Length);
                }
                else if (frame.Length > 0)
                {
                    process.StandardInput.BaseStream.Write(frame, 0, frame.Length);
                }
                else
                {
                    MemoryStream stream = new MemoryStream();

                    b.Save(stream, ImageFormat.Jpeg);

                    process.StandardInput.BaseStream.Write(stream.GetBuffer(), 0, stream.GetBuffer().Length);
                }
                process.StandardInput.Flush();
            }
            Time = Time.AddSeconds(1);
        }
    }
}
