﻿using Emgu.CV;
using Emgu.CV.Structure;
using ETC2019.Models;
using ETC2019.Models.Barcode.Linear;
using IronBarCode;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
// Program:
//   處理 QRCodePage 觸發動作。
// History:
// 2019/08/20 
namespace ETC2019.Controllers
{
    public class QRCodeController
    {
        private Configuration Config { get; set; }
        private DataModel[] Models { get; set; }
        /* 背景瀏覽資料夾，建立結構化物件。 */
        private BackgroundWorker Explorer { get; set; }
        public event ProgressChangedEventHandler ExplorerProgressChanged
        {
            add => Explorer.ProgressChanged += value;
            remove => Explorer.ProgressChanged -= value;
        }
        public event RunWorkerCompletedEventHandler ExplorerRunWorkerCompleted
        {
            add => Explorer.RunWorkerCompleted += value;
            remove => Explorer.RunWorkerCompleted -= value;
        }
        /* 背景掃描影像，修改結構化物件。 */
        private BackgroundWorker Scanner { get; set; }
        public event ProgressChangedEventHandler ScannerProgressChanged
        {
            add => Scanner.ProgressChanged += value;
            remove => Scanner.ProgressChanged -= value;
        }
        public event RunWorkerCompletedEventHandler ScannerRunWorkerCompleted
        {
            add => Scanner.RunWorkerCompleted += value;
            remove => Scanner.RunWorkerCompleted -= value;
        }
        /* 背景註記影像，並上傳資料庫。 */
        private BackgroundWorker Noter { get; set; }
        public event ProgressChangedEventHandler NoterProgressChanged
        {
            add => Noter.ProgressChanged += value;
            remove => Noter.ProgressChanged -= value;
        }
        public event RunWorkerCompletedEventHandler NoterRunWorkerCompleted
        {
            add => Noter.RunWorkerCompleted += value;
            remove => Noter.RunWorkerCompleted -= value;
        }
        /* 背景載入影像 */
        public Stack<string> ReaderStack { get; set; }
        private BackgroundWorker Reader { get; set; }
        public event RunWorkerCompletedEventHandler ReaderRunWorkerCompleted
        {
            add => Reader.RunWorkerCompleted += value;
            remove => Reader.RunWorkerCompleted -= value;
        }
        /* 回傳通知訊息 */
        public delegate void MessageShowEventHandler(string text, string caption, MessageBoxButtons buttons);
        public event MessageShowEventHandler MessageShow;
        /* 其他參數 */
        public Return[] Returns
        {
            get
            {
                List<Return> items = new List<Return>(Config.Return.Items);

                items.Sort();

                return items.ToArray();
            }
        }
        public bool IsBusy { get => Explorer.IsBusy || Scanner.IsBusy; }
        public string BookID { get; set; }
        public object ReturnID { get; set; }
        public string DepositDT { get; set; }
        public int Count { get => (from model in Models select model.Items.Length).ToArray().Sum(); }
        public string SystemText { get => string.Format("{0}v{1}", Config.SystemName, Config.SystemVersion); }

        public QRCodeController()
        {
            Config = File.Exists("config.json") ? JsonConvert.DeserializeObject<Configuration>(ReadToEnd("config.json")) : Configuration.Default();

            Config.SystemName = "ETC寄存退件註記工具";

            Config.SystemVersion = "2.2.3";

            Explorer = new BackgroundWorker() { WorkerReportsProgress = true, WorkerSupportsCancellation = true };

            Explorer.DoWork += Explorer_DoWork;

            Explorer.RunWorkerCompleted += Explorer_RunWorkerCompleted;

            Scanner = new BackgroundWorker() { WorkerReportsProgress = true, WorkerSupportsCancellation = true };

            Scanner.DoWork += Scanner_DoWork;

            Scanner.ProgressChanged += Scanner_ProgressChanged;

            Scanner.RunWorkerCompleted += Scanner_RunWorkerCompleted;

            Noter = new BackgroundWorker() { WorkerReportsProgress = true, WorkerSupportsCancellation = true };

            Noter.DoWork += Noter_DoWork;

            Noter.RunWorkerCompleted += Noter_RunWorkerCompleted;

            ReaderStack = new Stack<string>();

            Reader = new BackgroundWorker() { WorkerReportsProgress = true, WorkerSupportsCancellation = true };

            Reader.DoWork += Reader_DoWork;

            Reader.RunWorkerCompleted += Reader_RunWorkerCompleted;
        }
        // 讀取檔案的所有內容。
        private string ReadToEnd(string json)
        {
            using (StreamReader sr = new StreamReader("config.json"))
                return sr.ReadToEnd();
        }
        /// <summary>
        /// 開啟資料夾瀏覽器並載入資料夾中的檔案。
        /// </summary>
        /// <returns>選取的資料夾路徑。</returns>
        public string Explore()
        {
            FolderBrowserDialog folderBrowser = new FolderBrowserDialog()
            {
                RootFolder = Environment.SpecialFolder.Desktop,
                SelectedPath = string.IsNullOrEmpty(Config.SelectedPath) || !Directory.Exists(Config.SelectedPath) ? string.Empty : Config.SelectedPath,
            };
            if (!Explorer.IsBusy && folderBrowser.ShowDialog() == DialogResult.OK)
                Explorer.RunWorkerAsync(Config.SelectedPath = folderBrowser.SelectedPath);
            else if (Explorer.IsBusy)
                MessageShow(string.Format("正在載入：{0}，請稍後再試。", Config.SelectedPath), "選取資料夾失敗", MessageBoxButtons.OK);
            else
                MessageShow(string.Format("操作取消。", Config.SelectedPath), "選取資料夾失敗", MessageBoxButtons.OK);

            return Config.SelectedPath;
        }
        // 建立結構化檔案物件。
        private void Explorer_DoWork(object sender, DoWorkEventArgs e)
        {
            List<DataModel> models = new List<DataModel>();
            // 取得檔案清單
            List<string> f = new List<string>(GetFiles(e.Argument as string));

            f.Sort();

            int sum = 0;
            // 處理每一個檔案，建立結構化物件 DataModel。
            for (int i = 0; i < f.Count && !(sender as BackgroundWorker).CancellationPending; i++)
            {
                int percentProgress = Convert.ToInt32(Math.Ceiling(Convert.ToDouble(i) * 100 / f.Count));
                // 排除不為影像的檔案。
                if (Config.AllowExtension.Contains(Path.GetExtension(f[i])))
                {
                    DataModel model = new DataModel() { CreateDT = DateTime.Now, ModifyDT = DateTime.Now, Filename = f[i], Items = new Data[0] };

                    models.Add(model);

                    (sender as BackgroundWorker).ReportProgress(percentProgress, ++sum);
                }
            }
            e.Result = models.ToArray();
        }
        // 搜尋並建立檔案清單。
        private string[] GetFiles(string path)
        {
            List<string> files = new List<string>();
            // 搜尋資料夾
            foreach (string d in Directory.GetDirectories(path))
                files.AddRange(GetFiles(d));
            // 建立檔案清單
            foreach (string f in Directory.GetFiles(path))
                files.Add(f);
            // 回傳檔案清單
            return files.ToArray();
        }
        
        private void Explorer_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            Models = e.Result as DataModel[];
        }
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public bool Scan(int width, int height, out Bitmap frontImage)
        {
            Graphics g = Graphics.FromImage(frontImage = new Bitmap(width, height));

            g.DrawRectangles(new Pen(Color.Red, 4), Config.ROI.Mapping(width, height));

            g.Save();

            if (!Scanner.IsBusy)
                Scanner.RunWorkerAsync(Models);
            else
                return false;
            return true;
        }

        private void Scanner_DoWork(object sender, DoWorkEventArgs e)
        {
            DataModel[] models = e.Argument as DataModel[];

            SQL etcMetered = Config.SQL.Pairs["[nhpb].[dbo].[etcMetered]"];

            SQL etc = Config.SQL.Pairs["[nhpb].[dbo].[ETC寄存退件主檔_97]"];

            Code39 code = new Code39();

            long start = DateTime.Now.Ticks;

            int success = 0, failed = 0, sum = 0;

            for (int i = 0; i < models.Length; i++)
            {
                int percentProgress = Convert.ToInt32(Math.Ceiling(Convert.ToDouble(i) * 100 / models.Length));

                List<string> id = new List<string>();

                string text = code.Decode(models[i].Filename, Config.ROI);

                if (!string.IsNullOrEmpty(text))
                {
                    if (text.Length > 20 && int.TryParse(text.Substring(20, 2), out int count))
                    {
                        for (int j = 0; j < count; j++)
                            id.Add(text.Substring(22 + j * 9, 9));
                    }
                    else
                    {
                        string[] data = (from Data j in Find(text) select j.ID).ToArray();

                        if (data != null && data.Length > 0)
                            id.AddRange(data);
                    }
                }

                //(sender as BackgroundWorker).ReportProgress(percentProgress, id.ToArray());

                if (id.Count > 0)
                {
                    DataRowCollection rowMetered = etcMetered.Select(false, "tkt_no", id.ToArray());

                    DataRowCollection row = etc.Select(false, "tkt_no", id.ToArray());

                    models[i].Items = new Data[id.Count];

                    models[i].Items = (from j in id select new Data() { ID = j, ReturnNote = new Return() { ID = "-1", Text = "查無此案號" } }).ToArray();

                    for (int j = 0; j < rowMetered.Count; j++)
                    {
                        string tkt_no = rowMetered[j]["tkt_no"] as string;

                        models[i].Pairs[tkt_no].ReturnNote = new Return() { ID = "-1", Text = "未註記" };
                    }

                    for (int j = 0; j < row.Count; j++)
                    {
                        string tkt_no = row[j]["tkt_no"] as string;

                        string rsn = row[j]["rsn"] as string;

                        string thing = row[j]["thing"] as string;

                        models[i].Pairs[tkt_no].ReturnNote = thing.Equals("1") ? Config.Return.Pairs[rsn] : new Return() { ID = "-1", Text = "未註記" };
                    }

                    success = success + 1;

                    sum = sum + id.Count;

                    (sender as BackgroundWorker).ReportProgress(percentProgress, new KeyValuePair<string, bool>(Path.GetFileName(models[i].Filename), true));
                }
                else
                {
                    failed = failed + 1;

                    (sender as BackgroundWorker).ReportProgress(percentProgress, new KeyValuePair<string, bool>(Path.GetFileName(models[i].Filename), false));
                }

                (sender as BackgroundWorker).ReportProgress(percentProgress, new int[] { sum, success, failed });

                (sender as BackgroundWorker).ReportProgress(percentProgress, new TimeSpan(DateTime.Now.Ticks - start));
            }

            e.Result = models;
        }

        private void Scanner_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            
        }

        private void Scanner_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            Models = e.Result as DataModel[];
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public string[] Read(string id)
        {
            if (Get(id, out DataModel model) > -1)
            {
                if (!Reader.IsBusy)
                    Reader.RunWorkerAsync(model);
                else
                    ReaderStack.Push(id);

                return model.ToArray();
            }
            else
                return new string[0];
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <param name="firstID"></param>
        /// <returns></returns>
        public string[] Read(string id, out string firstID)
        {
            firstID = string.Empty;

            if (Get(id, out DataModel model) > -1)
            {
                if (!Reader.IsBusy)
                    Reader.RunWorkerAsync(model);
                else
                    ReaderStack.Push(id);

                firstID = model.FirstID ?? string.Empty;

                return model.ToArray();
            }
            else
                return new string[0];
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public int Get(string id)
        {
            for (int i = 0; i < Models.Length; i++)
                if (Models[i].Pairs.ContainsKey(id) || Path.GetFileName(Models[i].Filename).Equals(id))
                    return i;
            return -1;
        }

        private int Get(string id, out DataModel model)
        {
            model = null;

            for (int i = 0; i < Models.Length; i++)
                if (Models[i].Pairs.ContainsKey(id) || Path.GetFileName(Models[i].Filename).Equals(id))
                {
                    model = Models[i];

                    return i;
                }
            return -1;
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <param name="list"></param>
        /// <returns></returns>
        public int Locate(string id, string[] list)
        {
            for (int i = 0; i < list.Length; i++)
            {
                if (Get(list[i]) > Get(id))
                    return i;
            }
            return list.Length;
        }

        public string[] Locate(object[] list1, object[] list2)
        {
            List<string> list = new List<string>();

            list.AddRange((from item in list1 select Path.GetFileName(Models[Get(item as string)].Filename)).ToArray());

            list.AddRange((from item in list2 select Path.GetFileName(Models[Get(item as string)].Filename)).ToArray());

            list.Sort();

            return list.ToArray();
        }

        private DataModel GetModel(string id)
        {
            return Get(id, out DataModel model) > -1 ? model : null;
        }

        private void Reader_DoWork(object sender, DoWorkEventArgs e)
        {
            using (FileStream fs = new FileStream((e.Argument as DataModel).Filename, FileMode.Open))
            {
                MemoryStream ms = new MemoryStream();

                fs.CopyTo(ms);

                fs.Close();

                ms.Flush();

                ms.Position = 0;

                e.Result = new Bitmap(ms);
            }
        }

        private void Reader_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            if (ReaderStack.Count > 0 && Get(ReaderStack.Pop(), out DataModel model) > -1)
            {
                (sender as BackgroundWorker).RunWorkerAsync(model);

                ReaderStack.Clear();
            }
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="index"></param>
        /// <param name="id"></param>
        /// <returns></returns>
        public string Set(string id, string changed)
        {
            if (Get(id, out DataModel model) > -1)
                model.Pairs[id].ID = changed;
            else
                return id;

            return changed;
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public int Find(string id, string tkt_no)
        {
            if (Get(id, out DataModel model) > -1)
            {
                model.Items = Find(tkt_no);

                if (model.Items != null)
                {
                    SQL etc = Config.SQL.Pairs["[nhpb].[dbo].[ETC寄存退件主檔_97]"];

                    DataRowCollection row = etc.Select(true, "tkt_no", (from Data i in model.Items select i.ID as string).ToArray());

                    for (int i = 0; i < row.Count; i++)
                        if (model.Pairs.ContainsKey(row[i]["tkt_no"] as string))
                        {
                            string rsn = row[i]["rsn"] as string;

                            string thing = row[i]["thing"] as string;

                            model.Pairs[row[i]["tkt_no"] as string].ReturnNote = thing.Equals("1") ? Config.Return.Pairs[rsn] : new Return() { ID = "-1", Text = "未註記" };
                        }

                    return model.Items.Length;
                }
                else
                    return 0;
            }
            else
                return 0;
        }

        private Data[] Find(string tkt_no)
        {
            SQL etcMetered = Config.SQL.Pairs["[nhpb].[dbo].[etcMetered]"];

            DataRowCollection find = etcMetered.Select(false, "tkt_no", new string[] { tkt_no });

            if (find.Count == 1)
            {
                DataRowCollection rowMetered = etcMetered.Select(false, "post_no2", new string[] { find[0]["post_no2"] as string });

                return (from DataRow i in rowMetered where i["post_date"] as string == find[0]["post_date"] as string select new Data() { ID = i["tkt_no"] as string, ReturnNote = new Return() { ID = "-1", Text = "未註記" } }).ToArray();
            }
            else
                return new Data[0];
        }

        public int FindEmpty(int current)
        {
            for (int i = current; i < Models.Length; i++)
                if (Models[i].Items.Length == 0)
                    return i;
            return -1;
        }
        /// <summary>
        /// 
        /// </summary>
        public bool Note(object[] items)
        {
            if (!Noter.IsBusy)
                Noter.RunWorkerAsync((from object item in items
                                     let model = GetModel(item as string)
                                     select model).ToArray());
            else
                return false;
            return true;
        }

        private void Noter_DoWork(object sender, DoWorkEventArgs e)
        {
            DataModel[] models = e.Argument as DataModel[];

            SQL etcMetered = Config.SQL.Pairs["[nhpb].[dbo].[etcMetered]"];

            SQL etc = Config.SQL.Pairs["[nhpb].[dbo].[ETC寄存退件主檔_97]"];

            //int count = 1;

            for (int i = 0; i < models.Length; i++)
            {
                DataRowCollection rows = etcMetered.Select(false, "tkt_no", (from item in models[i].Items select item.ID).ToArray());

                int percentProgress = Convert.ToInt32(Math.Ceiling(Convert.ToDouble(i) * 100 / models.Length));

                string index = Path.GetFileNameWithoutExtension(models[i].Filename).Substring(9, 3);

                string key = Path.GetFileName(models[i].Filename);

                bool allNoted = true;

                //bool[] hadNote = (from item in models[i].Items select !item.ReturnNote.ID.Equals("-1")).ToArray();

                for (int j = 0; j < rows.Count /*&& !hadNote.Contains(true)*/; j++)
                {
                    string[] value = new string[]
                    {
                        string.Format("{0}{1}", DateTime.Now.Year - 1911, DateTime.Now.ToString("MMdd")),
                        string.Format("{0}{1}", BookID, index),
                        (ReturnID as Return).ID,        // 註記代號
                        rows[j]["tkt_no"] as string,    // 單號
                        "1",                            // 3 : 紅單+送達證 2 : 1 :
                        DepositDT,                      // 寄存日
                        rows[j]["accuse_no"] as string, // 郵局代號
                    };

                    DataRowCollection rowTemp = etc.SelectAnd(false, new Dictionary<string, string>()
                    {
                        { "tkt_no", rows[j]["tkt_no"] as string },
                        { "thing", "1" },
                    });

                    if (rowTemp.Count == 0 && etc.Insert(value))
                    {
                        models[i].Items[j].ReturnNote = ReturnID as Return;

                        allNoted = allNoted && true;
                    }
                    else if (rowTemp.Count > 0)
                        allNoted = allNoted /*&& rowTemp[0]["寄存日"] as string == DepositDT*/ && false;
                }

                if (allNoted)
                {
                    (sender as BackgroundWorker).ReportProgress(percentProgress, new KeyValuePair<string, bool>(key, true));

                    (sender as BackgroundWorker).ReportProgress(percentProgress, models[i].Items.Length);
                }
                else
                {
                    (sender as BackgroundWorker).ReportProgress(percentProgress, new KeyValuePair<string, bool>(key, false));
                }
                

                Thread.Sleep(50);
            }

            e.Result = models;
        }

        private void Noter_RunWorkerCompleted(object sender, RunWorkerCompletedEventArgs e)
        {
            //Models = e.Result as DataModel[];
        }

        public void Save()
        {
            Config.Save("");
        }
    }
}
